/**
 * 
 */
package gov.va.med.mhv.vitals.web.pgd;

import java.util.Date;
import java.util.UUID;

import javax.servlet.http.HttpSession;

import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Before;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.liferay.portal.kernel.events.ActionException;
import com.liferay.portal.kernel.log.Log;
import com.liferay.portal.kernel.log.LogFactoryUtil;
import com.liferay.portal.kernel.servlet.PortalSessionThreadLocal;

import gov.va.med.mhv.client.vamf.user.api.UserApi;
import gov.va.med.mhv.common.api.dto.PatientDTO;
import gov.va.med.mhv.common.api.dto.UserProfileDTO;
import gov.va.med.mhv.common.jwt.JwtUtility;

/**
 * @author DNS
 *
 */
@Aspect
@Component
public class VamfUserServiceAuthenticatorAspect {
	
	private static final Log LOG = LogFactoryUtil.getLog(VamfUserServiceAuthenticatorAspect.class);
	
	private static final String VAMF_JWT_TOKEN_ATTR_NAME = "MhvVitalsVamfAuthenticatedJwt";
	
	private static final String USER_PROFILE_DTO_ATTR_NAME = "LIFERAY_SHARED_userprofiledto";
	
	private static final String PATIENT_DTO_ATTR_NAME = "LIFERAY_SHARED_patientdto";
	
	@Autowired
	private ObjectMapper mapper;
	
	@Autowired
	private UserApi userApi;
	
	@Value("${vamf.mhv.issuer.id}")
	private String mhvIssuerId;
	
	@Value("${vamf.jwt.not.before.minutes}")
	private Integer notBeforeMinutes;
	
	@Value("${vamf.jwt.expire.after.minutes}")
	private Integer expireAfterMinutes;

	@Value("${vamf.mhv.signature.private.key}")
	private String mhvMobilePrivateKey;
	
	@Value("${vamf.mhv.authentication.authority}")
	private String mhvAuthenticationAuthority;

	@Before("@annotation(VamfJwtAuth)")
	public Object doAuthenticate(ProceedingJoinPoint joinPoint) throws Throwable {
		
		HttpSession session = null;
		
		try {
			JwtUtility jwtUtility = new JwtUtility(mhvIssuerId, notBeforeMinutes, expireAfterMinutes, mhvMobilePrivateKey);
			
			session = PortalSessionThreadLocal.getHttpSession();
			UserProfileDTO userProfile = getSafeRequestAttribute(session, USER_PROFILE_DTO_ATTR_NAME, UserProfileDTO.class);
			PatientDTO patient = getSafeRequestAttribute(session, PATIENT_DTO_ATTR_NAME, PatientDTO.class);
			String vamfAuthJwt = (String)session.getAttribute(VAMF_JWT_TOKEN_ATTR_NAME);
			
			// If the JWT is invalid, request / create a new token
			if(!jwtUtility.isTokenInvalid(vamfAuthJwt) && userProfile != null) {
				
				String firstName = userProfile.getFirstName();
				String middleName = userProfile.getMiddleName();
				String lastName = userProfile.getLastName();
				String ssn = userProfile.getSsn();
				String icn = (userProfile.getIsPatient() && patient != null) ? patient.getIcn() : "";
				String email = userProfile.getContact().getEmail();
				String gender = userProfile.getGender();
				Date birthDate = userProfile.getBirthDate();
				String mhvId = userProfile.getId().toString();
				
				// Create the MHV Signed web token.
				vamfAuthJwt = jwtUtility.createMobileWebToken(firstName, middleName, lastName, icn, ssn, email,
						gender, birthDate, mhvId, mhvAuthenticationAuthority, UUID.randomUUID().toString());
				
				// Make the request to the user service to get it signed by it
				vamfAuthJwt = this.userApi.createUserSessionFromJwt(vamfAuthJwt);
				
				session.setAttribute(VAMF_JWT_TOKEN_ATTR_NAME, vamfAuthJwt);
			}
			
		} catch (Throwable t) {
			LOG.error("Error establishing JWT Session.", t);
			throw t;
		}
		
		return joinPoint.proceed();
	}
	
	@SuppressWarnings("unchecked")
	private <T> T getSafeRequestAttribute(HttpSession session, String attributeName, Class<T> valueType)
			throws ActionException {
		try {
			Object sessionAttribute = session.getAttribute(attributeName);
			if (sessionAttribute != null && sessionAttribute instanceof String) {
				return mapper.readValue((String) sessionAttribute, valueType);
			}

			return (T) sessionAttribute;
		} catch (Exception e) {
			throw new ActionException(String.format("Error getting session attribute '%s' from session.", attributeName), e);
		}

	}
}
